<?php

namespace App\Http\Controllers\panel;

use App\Http\Controllers\Controller;
use App\Post;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Symfony\Component\HttpFoundation\Response;

class PostController extends Controller
{

    public function __construct()
    {
        $this->authorizeResource(Post::class);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $posts = Post::latest()->paginate(20);
        return view('panel.posts.list', compact('posts'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        return view('panel.posts.add');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'text' => 'required',
            'postType' => 'required|in:news,faq',
            'postStatus' => 'required|in:draft,public'
        ]);

        Post::create([
            'title' => $request->title,
            'content' => clean($request->text),
            'type' => $request->postType,
            'status' => $request->postStatus,
            'user_id' => auth()->user()->id
        ]);

        return redirect()->route('panel.posts.index')->withSuccess('با موفقیت اضافه شد');
    }

    /**
     * Display the specified resource.
     *
     * @param Post $post
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show(Post $post)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Post $post
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Post $post)
    {
        return view('panel.posts.edit', compact('post'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Post $post)
    {
        $request->validate([
            'title' => 'required',
            'text' => 'required',
            'postType' => 'required|in:news,faq',
            'postStatus' => 'required|in:draft,public'
        ]);

        $post->update([
            'title' => $request->title,
            'content' => clean($request->text),
            'type' => $request->postType,
            'status' => $request->postStatus,
        ]);

        return redirect()->back()->withSuccess('با موفقیت ویرایش شد');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    public function addMedia(Request $request)
    {
        $allowedMimeTypes = str_replace('.', '', config('upload_allowed_formats'));
        try {
            Validator::make($request->all(), [
                'file' => "mimes:$allowedMimeTypes"
            ])->validate();
        } catch (ValidationException $exception) {
            return response()->json($exception->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }


        $path = public_path('files/posts');

        if (!file_exists($path)) {
            mkdir($path, 0764, true);
        }

        $file = $request->file('file');

        $name = Str::random(64) . '_' . trim($file->getClientOriginalName());

        $file->move($path, $name);

        return response()->json([
            'location' => config('site_url') . '/files/posts/' . $name,
            'original_name' => $file->getClientOriginalName(),
        ]);
    }


    /**
     * show single news to user
     * @param Post $post
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function showSingleNews(Post $post)
    {
        return view('public.news-single', compact('post'));
    }
}
