<?php

namespace App;

use App\Events\UserRegistered;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class User extends Authenticatable implements HasMedia
{
    use Notifiable;
    use InteractsWithMedia;

    protected $guarded = [];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'mobile_verified_at' => 'datetime',
    ];

    public function setPasswordAttribute($password)
    {
        $this->attributes['password'] = bcrypt($password);
    }

    public function isVerifiedMobile(?User $user = null)
    {
        if ($user) {
            return $user->mobile_verified_at ?: false;
        }
        return $this->mobile_verified_at ?: false;
    }

    public function isVerifiedEmail(?User $user = null)
    {
        if ($user) {
            return $user->email_verified_at ?: false;
        }
        return $this->email_verified_at ?: false;
    }

    public function accountVerificationActions()
    {
        $mustVerifyMobile = config('must_verify_mobile');
        $mustVerifyEmail = config('must_verify_email');
        $verificationToken = Verification::createEmailToken();
        $request = request();

        if ($mustVerifyEmail and !$mustVerifyMobile) {
            $verification = Verification::create([
                'user_id' => $this->id,
                'type' => 'email',
                'ip' => $request->ip(),
                'token' => $verificationToken
            ]);
            //call user registered event so we can send email and SMS
            event(new UserRegistered($this, $verification, null));
        } elseif ($mustVerifyMobile and !$mustVerifyEmail) {
            $verification = Verification::create([
                'user_id' => $this->id,
                'type' => 'sms',
                'ip' => $request->ip(),
                'token' => Verification::createMobileToken()
            ]);

            //call user registered event so we can send email and SMS
            event(new UserRegistered($this, $verification, null));
        } else {
            $emailVerification = Verification::create([
                'user_id' => $this->id,
                'type' => 'email',
                'ip' => $request->ip(),
                'token' => $verificationToken
            ]);

            $mobileVerification = Verification::create([
                'user_id' => $this->id,
                'type' => 'sms',
                'ip' => $request->ip(),
                'token' => Verification::createMobileToken()
            ]);

            //call user registered event so we can send email and SMS
            event(new UserRegistered($this, $emailVerification, $mobileVerification));
        }
    }

    public function isUser()
    {
        return $this->role == 'user';
    }

    public function isStaff()
    {
        return $this->role == 'staff';
    }

    public function isAdmin()
    {
        return $this->role == 'admin' or $this->role == 'superadmin';
    }

    public function isSuperAdmin()
    {
        return $this->role == 'superadmin';
    }

    public function isSuperior()
    {
        if ($this->isAdmin() or $this->isStaff()) {
            return true;
        }
        return false;
    }

    public function department()
    {
        return $this->belongsToMany(Department::class);
    }

    public function replies()
    {
        return $this->hasMany(Reply::class, 'replied_by', 'id');
    }

    public function isSuperiorStaff()
    {
        $replies = Reply::orderByRate()->take(5)->get()->toArray();
        $superiorStaffs = array_column($replies, 'user_id');
        if (in_array($this->id, $superiorStaffs)) {
            return true;
        }
        return false;
    }
}
